/******************************************************************************
 * 	qam - a qt 'abfahrtsmonitor
 *	Copyright (C) 2007  gf@nichtcool.de
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version 2
 *	of the License.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 *	USA.
 *****************************************************************************/

#include "monitor.h"

#include <QMessageBox>
#include <QByteArray>
#include <QTextStream>

#include "options.h"

#ifdef _DEBUG
	#include <iostream>
#endif


CMonitor::CMonitor(QWidget* parent, Qt::WFlags f) :	
	QWidget(parent, f),
	m_topLayout(this),
	m_Http(this),
	m_nHttpRequestId(0),
	m_bHttpRequestFailed(false),
	m_nUpdateInterval(10),
	m_location("Dresden"),
	m_station("Albertplatz"),
	m_eta("3")
{
	// init widgets
	
	m_topLayout.setMargin(0);
	m_topLayout.setSpacing(0);
	m_topLayout.addWidget(&m_tabWidget);

	m_tabWidget.setTabPosition(QTabWidget::South);
	m_tabWidget.addTab(&m_pageMonitor, "Monitor");
	m_tabWidget.addTab(&m_pageSetMonitor, "ndern");
	m_tabWidget.addTab(&m_pageConfigure, "Einstellungen");

	// load main window state

	loadSettings();

	COptions& settings = COptions::getInstance();
	settings.beginGroup("MainWindow");
		resize(settings.value("size", QSize(300, 200)).toSize());
		move(settings.value("pos", QPoint(200, 200)).toPoint());
	settings.endGroup();

	// set up event handlers

	connect(&m_Http, SIGNAL(requestFinished(int, bool)),
			this, SLOT(onHttpRequestFinished(int, bool)));
	connect(&m_Http, SIGNAL(responseHeaderReceived(const QHttpResponseHeader &)),
                this, SLOT(onResponseHeaderReceived(const QHttpResponseHeader &)));
	connect(&COptions::getInstance(), SIGNAL(valuesChanged(QStringList&, void*)),
				this, SLOT(onValuesChanged(QStringList&, void*)));

	// start retrieving connection data
	
	getNewConnections();

	// start refresh timer

	connect(&m_timer, SIGNAL(timeout()), this, SLOT(onTimer()));
	m_timer.start(m_nUpdateInterval * 1000);
}

CMonitor::~CMonitor()
{

}

bool CMonitor::getNewConnections()
{
	QString qstrUrl;
	qstrUrl = "/abfahrtsmonitor/Abfahrten.do?";
	qstrUrl += "ort="; qstrUrl += m_location;
	qstrUrl += "&hst="; qstrUrl += m_station;
	qstrUrl += "&vz="; qstrUrl += m_eta;

	m_Http.setHost("widgets.vvo-online.de", 80);
	m_nHttpRequestId = m_Http.get(qstrUrl);

	return true;
}

void CMonitor::onResponseHeaderReceived(const QHttpResponseHeader &responseHeader)
{
    if (responseHeader.statusCode() != 200) {
        QMessageBox::information(this, tr("HTTP"), tr("Download failed: %1.").arg(responseHeader.reasonPhrase()));
        m_bHttpRequestFailed = true;
        m_Http.abort();
        return;
    }
}

void CMonitor::onHttpRequestFinished(int requestId, bool error)
{
	if(error || m_bHttpRequestFailed)
		return;

	// extract body

	QByteArray res = m_Http.readAll();
	if(res.length() == 0)
		return;
	QTextStream stream(res);
	stream.setCodec("UTF-8");
	stream.setAutoDetectUnicode(true);

	QString qstr;
	QChar ch;

	while(!stream.atEnd()) {
		stream >> ch;
		qstr += ch;
	}

	// body empty?

	if(qstr.length() == 0)
		return;

	qstr.replace("&quot;", "\"");

	// decode html-escaped characters

	QString qstrTmp;
	int nTmp;
	int pos = qstr.indexOf("&#");
	int end;

	while(pos != -1) {
		end = pos + 1;
		
		while(qstr[end] != ';')
			end++;

		if(end < (pos + 3))
			break;

		qstrTmp = qstr.mid(pos + 2, end - pos - 2);
		//QMessageBox::information(this, "extracted sub string", qstrTmp);

		nTmp = qstrTmp.toInt();
		QChar qchar(nTmp);
		qstr.replace(pos, end - pos + 1, qchar);
		
		pos = qstr.indexOf("&#");
	}

	//QMessageBox::information(this, "request result", qstr);

	// extract connection data

	QList<ConnectionEntry> connections;
	QStringList strList;

	qstr.remove(0, 2);
	qstr.remove(qstr.length() - 2, 2);
	strList = qstr.split("],[", QString::SkipEmptyParts);
	if(strList.isEmpty()) {
		// ... error
		return;
	}

	QString conn;
	QString connEl;
	QStringList strList2;
	ConnectionEntry entry;

	while(!strList.isEmpty()) {
		conn = strList.takeFirst();
		conn.remove(0, 1); // remove leading '"'
		conn.remove(conn.length() - 1, 1); // and trailing '"'
		strList2 = conn.split("\",\""); // split n","richtung","n into [n][richtung][n]
		
		if(strList2.size() != 3)
			continue;

		entry.ident = strList2.at(0);
		entry.direction = strList2.at(1);
		entry.eta = strList2.at(2);
		
		connections.append(entry);
	}

#ifdef _DEBUG
	std::cout << "++++++++++++++\n";
	for(pos = 0 ; pos < connections.size(); pos++) {
		std::cout << (const char*)connections.at(pos).ident.toLatin1() << ", " << (const char*)connections.at(pos).direction.toLatin1() << ", " << (const char*)connections.at(pos).eta.toLatin1() << "\n";
	}
#endif

	m_pageMonitor.Update(connections);
}

void CMonitor::onTimer()
{
	if(m_Http.currentId() == 0) {
		getNewConnections();
	}

	if(m_timer.interval() != m_nUpdateInterval) {
		m_timer.setInterval(m_nUpdateInterval * 1000);
	}
}

void CMonitor::onButtonMonitorClicked()
{

}

void CMonitor::onButtonSetMonitorClicked()
{

}

void CMonitor::onButtonConfigureClicked()
{

}

void CMonitor::closeEvent(QCloseEvent *event)
{	
	m_timer.stop();
	writeSettings();
	// these should be only written on shutdown ;)
	COptions& settings = COptions::getInstance();
	settings.beginGroup("MainWindow");
		settings.setValue("size", size());
		settings.setValue("pos", pos());
	settings.endGroup();
	event->accept();
}

void CMonitor::loadSettings()
{
	COptions& settings = COptions::getInstance();
	settings.beginGroup("Monitor");
		m_location = settings.value("location", "Dresden").toString();
		m_station = settings.value("station", "Albertplatz").toString();
		m_eta = settings.value("eta", "3").toString();
	settings.endGroup();
	settings.beginGroup("MainWindow");
		m_nUpdateInterval = settings.value("updateInterval", 10).toInt();
		setWindowOpacity(settings.value("windowOpacity", 0.8).toDouble());
		Qt::WindowFlags flags = windowFlags();
#if 0
		if(settings.value("alwaysOnTop", true).toBool())
			flags |= Qt::WindowStaysOnTopHint;
		else
			flags ^= Qt::WindowStaysOnTopHint;
		setWindowFlags(flags);
#endif
    settings.endGroup();
}

void CMonitor::writeSettings()
{
	COptions& settings = COptions::getInstance();
	settings.beginGroup("Monitor");
		settings.setValue("location", m_location);
		settings.setValue("station", m_station);
		settings.setValue("eta", m_eta);
	settings.endGroup();
	settings.beginGroup("MainWindow");
		settings.setValue("updateInterval", m_nUpdateInterval);
		settings.setValue("windowOpacity", windowOpacity());
#if 0
		Qt::WindowFlags flags = windowFlags();
		settings.setValue("alwaysOnTop", (flags & Qt::WindowStaysOnTopHint) ? true : false);
#endif
    settings.endGroup();

	QStringList list("MainWindow");
	list += "Monitor";
	settings.adviseOfChanges(list, (void*)this);
}

void CMonitor::onValuesChanged(QStringList& affected, void* causer)
{
	if((causer != (void*)this) && (affected.contains("MainWindow") || affected.contains("Monitor"))) {
		loadSettings();
	}
}